from django.contrib.auth.models import User
from django.db import models
from django.utils import timezone
from io import BytesIO
from PIL import Image
from django.core.files import File
import uuid, time

BASE_URL = 'http://127.0.0.1:8000'

def compress(image):
    im = Image.open(image)
    rgb_im = im.convert("RGB")
    im_io = BytesIO()
    rgb_im.save(im_io, format = im.format, quality = 50)
    new_image = File(im_io, name=image.name)
    return new_image

def make_thumbnail(image, size=(300, 200)):
    img = Image.open(image)
    img.convert('RGB')
    img.thumbnail(size)
    thumb_io = BytesIO()
    img.save(thumb_io, 'JPEG', quality=85)
    thumbnail = File(thumb_io, name=image.name)
    return thumbnail

def gen_uid():
    return f"{uuid.uuid4()}-{str(time.time).replace('.', '-')}"

class UserProfile(models.Model):
    user = models.OneToOneField(User, on_delete=models.CASCADE)
    address = models.TextField()
    postal_code = models.CharField(max_length=10)
    image = models.ImageField(upload_to ='profile_image_uploads/', blank=True, null=True)
    thumbnail = models.ImageField(upload_to ='profile_image_uploads/', blank=True, null=True)
    uid = models.CharField(max_length=120, default = gen_uid())
    is_active = models.BooleanField(default=True)
    datetime_created = models.DateTimeField(default=timezone.now)
    datetime_updated = models.DateTimeField(auto_now=True)

    class Meta:
         verbose_name_plural = 'Profile'
         ordering = ('user',)

    def __str__(self):
        return f"<User Profile [{self.user}]>"

    def save(self, *args, **kwargs):
        new_image = compress(self.image)
        self.image = new_image
        super().save(*args, **kwargs)
    
    def get_absolute_url(self):
        return f'/profile/{self.pk}/'
    
    def get_image(self):
        if self.image:
            return BASE_URL + self.image.url
        return ''
    

    def get_thumbnail(self):
        if self.thumbnail:
            return BASE_URL + self.thumbnail.url
        else:
            if self.image:
                self.thumbnail = make_thumbnail(self.image)
                self.save()

                return BASE_URL + self.thumbnail.url
            else:
                return f'/static/assets/gif/link.gif'
            

    """
    def make_thumbnail(self, image, size=(300, 200)):
        img = Image.open(image)
        img.convert('RGB')
        img.thumbnail(size)
        
        thumb_io = BytesIO()
        img.save(thumb_io, 'JPEG', quality=85)
        thumbnail = File(thumb_io, name=image.name)

        return thumbnail
    """

