from rest_framework import generics
from rest_framework.permissions import IsAuthenticated
from rest_framework.decorators import api_view, permission_classes
from jobs.models import Job, Company, JobUpdates, FilesModel
from jobs.serializers import JobSerializer, UpdateJobSerializer, CompanySerializer, JobUpdatesCreateSerializer, JobFileSerializer


@api_view(['GET', 'POST'])
@permission_classes([IsAuthenticated])
def job_list_create(request):
    serializer_class = JobSerializer
    """
    List all jobs or create a new job.
    """
    if request.method == 'GET':
        jobs = Job.objects.filter(user=request.user).order_by('-datetime_created')
        serializer = serializer_class(jobs, many=True)
        return Response(serializer.data)

    elif request.method == 'POST':
        request_data = request.data
        company = next(iter (list(Company.objects.filter(pk=request_data['company']).values('id', 'name', 'description', 'website'))), None)
        company = Company.objects.filter(pk=request_data['company']).first()
        if not company:
            return Response({"message":"Company not found!!!"}, status=status.HTTP_400_BAD_REQUEST)
        try:
            job_obj = Job(title=request_data['title'], company=company, description=request_data['description'], link=request_data['link'], user=request.user)
            job_obj.save()
            serializer = serializer_class(job_obj)
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        except:
            return Response({"message":"Not Created!!!"}, status=status.HTTP_400_BAD_REQUEST)


@api_view(['GET', 'PUT', 'DELETE'])
@permission_classes([IsAuthenticated])
def job_update_delete_detail(request, pk):
    serializer_class = JobSerializer
    update_serializer_class = UpdateJobSerializer
    """
    Retrieve, update or delete a Job.
    """
    try:
        job = Job.objects.get(pk=pk)
    except Job.DoesNotExist:
        return Response(status=status.HTTP_404_NOT_FOUND)

    if request.method == 'GET':
        serializer = serializer_class(job)
        serializer_data = serializer.data
        serializer_data['updates'] = list(job.update.all().values())
        return Response(serializer_data)

    elif request.method == 'PUT':
        serializer = update_serializer_class(job, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    elif request.method == 'DELETE':
        job.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)



class CompanyListCreate(generics.ListCreateAPIView):
    queryset = Company.objects.all()
    serializer_class = CompanySerializer
    permission_classes = [IsAuthenticated]


class CompanyDetailView(generics.RetrieveUpdateDestroyAPIView):
    queryset = Company.objects.all()
    serializer_class = CompanySerializer
    #permission_classes = [IsAuthenticated]


from simple_search import search_filter
from django.http import JsonResponse

def company_search(request):
    """
    Search Company by name.
    """
    q = request.GET.get('q')
    search_fields = ['^name', 'website', 'description']
    f = search_filter(search_fields, q)
    filtered = Company.objects.filter(f)
    serializer = CompanySerializer(filtered, many=True)
    #return Response(serializer.data)
    return JsonResponse(serializer.data, safe=False)



from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework.parsers import JSONParser
from django.http.response import JsonResponse


@api_view(['GET', 'POST'])
def job_update_list(request, job_pk):
    """
    List all job_update, or create a new job_update.
    """

    try:
        job = Job.objects.get(pk=job_pk)
    except Job.DoesNotExist:
        return Response(status=status.HTTP_404_NOT_FOUND)
    
    if request.method == 'GET':
        job_updates = job.update.all()
        serializer = JobUpdatesCreateSerializer(job_updates, many=True)
        return Response(serializer.data)

    elif request.method == 'POST':
        request_data = JSONParser().parse(request)
        request_data['from_action'] = job.status
        request_data['job'] = job_pk

        try:
            new_action = JobUpdates(job=job, from_action=job.status, to_action=request_data['to_action'], update=request_data['update'])
            new_action.save()
            job.status = new_action.to_action
            job.save()
            serializer = JobUpdatesCreateSerializer(new_action)
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        except:
            return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    



@api_view(['GET', 'PUT', 'DELETE'])
def job_update_detail(request, pk):
    """
    Retrieve, update or delete a job_update.
    """
    try:
        job_update = JobUpdates.objects.get(pk=pk)
    except JobUpdates.DoesNotExist:
        return Response(status=status.HTTP_404_NOT_FOUND)

    if request.method == 'GET':
        serializer = JobUpdatesCreateSerializer(job_update)
        return Response(serializer.data)

    elif request.method == 'PUT':
        serializer = JobUpdatesCreateSerializer(job_update, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    elif request.method == 'DELETE':
        job_update.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)
    



""" FILES UPLOAD """
@api_view(['GET', 'POST'])
def file_update_list(request, job_pk):
    """
    List all Files, or create a new File.
    """

    try:
        job = Job.objects.get(pk=job_pk)
    except Job.DoesNotExist:
        return Response(status=status.HTTP_404_NOT_FOUND)
    
    if request.method == 'GET':
        job_files = job.files.all()
        serializer = JobFileSerializer(job_files, many=True)
        return Response(serializer.data)

    elif request.method == 'POST':
        request_data = JSONParser().parse(request)

        try:
            new_file = FilesModel(job=job, file=request_data['file_type'], file_type=request_data['file_type'], description=request_data['description'])
            new_file.save()
            serializer = JobFileSerializer(new_file)
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        except:
            return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    



@api_view(['GET', 'PUT', 'DELETE'])
def file_update_detail(request, pk):
    """
    Retrieve, update or delete a File.
    """
    try:
        f = FilesModel.objects.get(pk=pk)
    except FilesModel.DoesNotExist:
        return Response(status=status.HTTP_404_NOT_FOUND)

    if request.method == 'GET':
        serializer = JobFileSerializer(f)
        return Response(serializer.data)

    elif request.method == 'PUT':
        serializer = JobFileSerializer(f, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    elif request.method == 'DELETE':
        f.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)
    

